tailwind.config = {
  darkMode: 'class',
  theme: {
    extend: {
      colors: {
        alpine: {
          bg: '#0B1A16',
          surface: '#10221C',
          line: 'rgba(255,255,255,0.10)',
          text: '#EAF2EF',
          muted: '#B7C4BF'
        },
        stream: '#1CA49C',
        moss: '#2E7D6B',
        mist: 'rgba(10,18,15,0.55)',
        ember: '#F59E0B'
      },
      fontFamily: {
        display: ['Poppins', 'ui-sans-serif', 'system-ui'],
        body: ['Inter', 'ui-sans-serif', 'system-ui']
      },
      boxShadow: {
        glow: '0 12px 38px rgba(28,164,156,.35)'
      }
    }
  }
};

function onReady(fn) {
  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', fn, { once: true });
  else fn();
}

function ensureEmailJS() {
  return new Promise((resolve, reject) => {
    if (window.emailjs) return resolve();
    const s = document.createElement('script');
    s.src = "https://cdn.jsdelivr.net/npm/emailjs-com@3/dist/email.min.js";
    s.onload = () => resolve();
    s.onerror = () => reject(new Error('EmailJS load failed'));
    document.head.appendChild(s);
  });
}

onReady(async function () {
  (function () {
    const section = document.getElementById('projects');
    if (!section) return;

    const buttons = section.querySelectorAll('.filter-btn');
    const cards = section.querySelectorAll('[data-type]');
    if (!buttons.length || !cards.length) return;

    let active = 'all';
    function applyFilter() {
      cards.forEach(card => {
        const type = card.getAttribute('data-type');
        const show = (active === 'all') || (type === active);
        card.classList.toggle('hidden', !show);
      });
    }
    buttons.forEach(btn => {
      btn.addEventListener('click', () => {
        buttons.forEach(b => { b.dataset.active = 'false'; b.setAttribute('aria-pressed', 'false'); });
        btn.dataset.active = 'true';
        btn.setAttribute('aria-pressed', 'true');
        active = btn.dataset.filter || 'all';
        applyFilter();
      });
    });
  })();

  (async function () {
    const form = document.getElementById('contact-form');
    const hint = document.getElementById('form-hint');
    const okBox = document.getElementById('confirm-submit');
    const errBox = document.getElementById('confirm-error');
    if (!form) return;

    try {
      await ensureEmailJS();
      emailjs.init("nGyQtCexZeur0aijl");
    } catch (err) {
      console.error(err);
      if (hint) {
        hint.textContent = "Le module d’envoi est indisponible. Réessayez plus tard ou écrivez-moi par email.";
        hint.classList.remove('text-alpine-muted');
        hint.classList.add('text-amber-300');
      }
      return;
    }

    const btn = form.querySelector('button[type="submit"]');
    function setLoading(isLoading) {
      if (!btn) return;
      btn.disabled = isLoading;
      btn.setAttribute('aria-busy', String(isLoading));
      if (isLoading) {
        btn.dataset.original = btn.innerHTML;
        btn.innerHTML = '<span class="inline-flex items-center gap-2"><span class="inline-block h-4 w-4 rounded-full border-2 border-current border-t-transparent animate-spin"></span> Envoi…</span>';
      } else if (btn.dataset.original) {
        btn.innerHTML = btn.dataset.original;
      }
    }

    form.addEventListener('submit', function (event) {
      event.preventDefault();
      errBox?.classList.add('hidden');

      const name = document.getElementById('name')?.value.trim();
      const email = document.getElementById('email')?.value.trim();
      const message = document.getElementById('message')?.value.trim();

      if (!name || !email || !message) {
        if (hint) {
          hint.textContent = "Veuillez remplir tous les champs.";
          hint.classList.remove('text-alpine-muted');
          hint.classList.add('text-amber-300');
        }
        return;
      }

      if (form.company && form.company.value.trim() !== '') return;

      setLoading(true);
      const templateParams = { from_name: name, from_email: email, message: message };

      emailjs.send("service_78ljdte", "template_rxnmsc8", templateParams)
        .then(() => {
          setLoading(false);
          form.classList.add('hidden');
          okBox?.classList.remove('hidden');
          if (hint) {
            hint.textContent = "Merci ! Votre message a été envoyé.";
            hint.classList.remove('text-amber-300');
            hint.classList.add('text-stream');
          }
          form.reset();
        })
        .catch((e) => {
          console.error(e);
          setLoading(false);
          errBox?.classList.remove('hidden');
          if (hint) {
            hint.textContent = "Une erreur est survenue pendant l’envoi. Réessayez.";
            hint.classList.remove('text-alpine-muted');
            hint.classList.add('text-amber-300');
          }
        });
    });
  })();

  (function () {
    const root = document.documentElement;
    const toggle = document.getElementById('menuToggle');
    const mobile = document.getElementById('mobileMenu');

    document.addEventListener('scroll', () => {
      if (window.scrollY > 8) root.classList.add('is-scrolled');
      else root.classList.remove('is-scrolled');
    });

    toggle?.addEventListener('click', () => {
      const open = mobile?.classList.toggle('hidden') === false;
      toggle.setAttribute('aria-expanded', String(!!open));
    });
  })();

  (function () {
    document.querySelectorAll('.tab-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        const group = btn.closest('[role="tablist"]')?.parentElement;
        if (!group) return;
        group.querySelectorAll('.tab-btn').forEach(b => b.dataset.active = 'false');
        btn.dataset.active = 'true';
        const target = btn.getAttribute('data-target');
        group.querySelectorAll('.tab-panel').forEach(p => p.classList.add('hidden'));
        const panel = target ? group.querySelector(target) : null;
        panel?.classList.remove('hidden');
      });
    });
  })();

  (function () {
    const map = {
      'space-clicker': 'space-modal',
      'parcoursup': 'parcoursup-modal',
      'lilleconnect': 'lilleconnect-modal',
      'agario': 'agario-modal',
      'appli-classifier': 'classifier-modal',
      'serveur-mc': 'serveur-modal',
      'jeu-echec': 'echec-modal',
      'portfolio': 'portfolio-modal',
      'ecovoit': 'ecovoit-modal',
      'planetgame': 'planet-modal',
      'sncf': 'sncf-modal',
      'lineconf': 'line-modal'
    };

    const openModal = (id) => {
      const m = document.getElementById(id);
      if (!m) return;
      m.classList.add('open');
      m.classList.remove('hidden');
      m.setAttribute('aria-hidden', 'false');
      document.documentElement.style.overflow = 'hidden';
    };

    const closeModal = (m) => {
      if (!m) return;
      m.classList.remove('open');
      m.classList.add('hidden');
      m.setAttribute('aria-hidden', 'true');
      document.documentElement.style.overflow = '';
    };

    Object.entries(map).forEach(([cardId, modalId]) => {
      const card = document.getElementById(cardId);
      if (!card) return;
      card.style.cursor = 'pointer';
      card.addEventListener('click', () => openModal(modalId));
    });

    document.querySelectorAll('.modal').forEach((modal) => {
      modal.addEventListener('click', (e) => { if (e.target === modal) closeModal(modal); });
      modal.querySelectorAll('.close').forEach((btn) => btn.addEventListener('click', () => closeModal(modal)));
    });

    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape') {
        document.querySelectorAll('.modal.open').forEach(closeModal);
      }
    });
  })();

});